/*
 * control.c
 *
 *  Created on: Apr 6, 2015
 *      Author: Jordan
 */

#include "sys_common.h"
#include "config.h"
#include "guidance.h"
#include "control.h"
#include "PWM.h"
#include "stateflow.h"
#include "navigation.h"
#include "eeprom.h"

float32 vel_x_ref, vel_y_ref, vel_z_ref, roll_ref, pitch_ref, roll_rate_ref, pitch_rate_ref, yaw_rate_ref, thr_ref;
float32 m1_cmd, m2_cmd, m3_cmd, m4_cmd;

PID_controller_t roll_rate_PID, pitch_rate_PID, yaw_rate_PID, roll_PID, pitch_PID, vel_x_PID, vel_y_PID, vel_z_PID;
float32 PID_params[15] = {DEF_ROLL_RATE_P, DEF_ROLL_RATE_I, DEF_ROLL_RATE_D, DEF_PITCH_RATE_P, DEF_PITCH_RATE_I, DEF_PITCH_RATE_D, DEF_YAW_RATE_P, DEF_YAW_RATE_I, DEF_YAW_RATE_D, DEF_ROLL_P, DEF_ROLL_I, DEF_ROLL_D, DEF_PITCH_P, DEF_PITCH_I, DEF_PITCH_D};

void read_PID_params(){
	uint8 param_array_bytes[60];
	uint8 i;
	eeprom_read_array(PID_PARAMS_ADDR, 60, param_array_bytes);
	for(i=0;i<15;i++){
		memcpy(PID_params + i, param_array_bytes + i*4, 4);
	}
}

void write_PID_params(){
	uint8 param_array_bytes[60];
	uint8 i;
	for(i=0;i<15;i++){
		memcpy(param_array_bytes + i*4, PID_params + i, 4);
	}
	eeprom_write_array(PID_PARAMS_ADDR, 60, param_array_bytes);
}

void adjust_PID(PID_params_t PID, float32 value){
	PID_params[PID] += value;
	PID_params[PID] = PID_params[PID] < 0 ? 0 : PID_params[PID];
}

void set_vel_x_ref(float32 cmd){
	vel_x_ref = cmd;
}

void set_vel_y_ref(float32 cmd){
	vel_y_ref = cmd;
}

void set_vel_z_ref(float32 cmd){
	vel_z_ref = cmd;
}

void set_roll_ref(float32 cmd){
	roll_ref = cmd;
}

void set_pitch_ref(float32 cmd){
	pitch_ref = cmd;
}

void set_roll_rate_ref(float32 cmd){
	roll_rate_ref = cmd;
}

void set_pitch_rate_ref(float32 cmd){
	pitch_rate_ref = cmd;
}

void set_yaw_rate_ref(float32 cmd){
	yaw_rate_ref = cmd;
}

void set_thr_ref(float32 cmd){
	thr_ref = cmd;
}

void control_reset(){
	init_PID(&roll_rate_PID, 1.0/FS, PID_params[ROLL_RATE_P], PID_params[ROLL_RATE_I], PID_params[ROLL_RATE_D], ROLL_RATE_N);
	init_PID(&pitch_rate_PID, 1.0/FS, PID_params[PITCH_RATE_P], PID_params[PITCH_RATE_I], PID_params[PITCH_RATE_D], PITCH_RATE_N);
	init_PID(&yaw_rate_PID, 1.0/FS, PID_params[YAW_RATE_P], PID_params[YAW_RATE_I], PID_params[YAW_RATE_D], YAW_RATE_N);
	init_PID(&roll_PID, 1.0/FS, PID_params[ROLL_P], PID_params[ROLL_I], PID_params[ROLL_D], ROLL_N);
	init_PID(&pitch_PID, 1.0/FS, PID_params[PITCH_P], PID_params[PITCH_I], PID_params[PITCH_D], PITCH_N);
	init_PID(&vel_x_PID, 1.0/FS, VEL_X_P, VEL_X_I, VEL_X_D, VEL_X_N);
	init_PID(&vel_y_PID, 1.0/FS, VEL_Y_P, VEL_Y_I, VEL_Y_D, VEL_Y_N);
	init_PID(&vel_z_PID, 1.0/FS, VEL_Z_P, VEL_Z_I, VEL_Z_D, VEL_Z_N);

	reset_wpt_counter();
}

//float32 ver_roll[407] = {0.0000000000,0.0000000000,0.0000015130,0.0000107407,0.0000307083,0.0000616308,0.0001022579,0.0001506750,0.0002048341,0.0002628622,0.0003232113,0.0003847057,0.0004465202,0.0005081274,0.0005692327,0.0006297087,0.0006895344,0.0007487554,0.0008074524,0.0008657182,0.0009236426,0.0009813076,0.0010387910,0.0010961568,0.0011534540,0.0012107182,0.0012679744,0.0013252396,0.0013825246,0.0014398358,0.0014971770,0.0015545499,0.0016119501,0.0016693557,0.0017267426,0.0017840901,0.0018413805,0.0018985991,0.0019557336,0.0020127699,0.0020696892,0.0021264615,0.0021830410,0.0022393627,0.0022953436,0.0023508877,0.0024058865,0.0024602511,0.0025139159,0.0025668081,0.0026189330,0.0026706189,0.0027222630,0.0027741740,0.0028265505,0.0028795967,0.0029335635,0.0029887585,0.0030456109,0.0031047662,0.0031670005,0.0032332197,0.0033045227,0.0033820261,0.0034659362,0.0035528608,0.0036383332,0.0037179868,0.0037877663,0.0038436361,0.0038814414,0.0038966964,0.0038865013,0.0038542088,0.0038045671,0.0037421482,0.0036610694,0.0035174206,0.0032667167,0.0028850883,0.0023965560,0.0018378139,0.0012354281,0.0006318866,0.0001363295,-0.0001963319,-0.0003601359,-0.0003997024,-0.0004465950,-0.0005894096,-0.0008721064,-0.0013107403,-0.0018988148,-0.0026028794,-0.0033882763,-0.0042335789,-0.0051237019,-0.0060436839,-0.0069669717,-0.0078686964,-0.0087400503,-0.0096105332,-0.0104581962,-0.0112447646,-0.0119247193,-0.0124970932,-0.0130095178,-0.0135227487,-0.0141342975,-0.0148682455,-0.0155468179,-0.0161098466,-0.0165354382,-0.0167925605,-0.0167947920,-0.0165170681,-0.0160053541,-0.0153963227,-0.0147799650,-0.0142045765,-0.0136924792,-0.0132509372,-0.0128790475,-0.0125721883,-0.0123249158,-0.0121325440,-0.0119916679,-0.0119009174,-0.0118597909,-0.0118678224,-0.0119219196,-0.0120187590,-0.0122078840,-0.0125152581,-0.0129557526,-0.0135404339,-0.0142899588,-0.0152314006,-0.0164169293,-0.0179083321,-0.0197671646,-0.0220688033,-0.0249958468,-0.0287776956,-0.0337184430,-0.0408011555,-0.0510440157,-0.0651638193,-0.0839471340,-0.1077554322,-0.1368280868,-0.1711773821,-0.2115252997,-0.2582360394,-0.3107805068,-0.3680614700,-0.4288071349,-0.4918039659,-0.5559999153,-0.6205514111,-0.6847866044,-0.7482632622,-0.8107292960,-0.8721100013,-0.9324476607,-0.9919095724,-1.0506201711,-1.1086024937,-1.1659313035,-1.2227825970,-1.2793175899,-1.3356584708,-1.3918177082,-1.4477381094,-1.5033479126,-1.5586009501,-1.6135029646,-1.6681169878,-1.7225558802,-1.7769866904,-1.8316482909,-1.8867399492,-1.9424229510,-1.9988114601,-2.0559234259,-2.1137854995,-2.1724716700,-2.2320884957,-2.2927590442,-2.3546103559,-2.4176800831,-2.4816072563,-2.5451746953,-2.6056328721,-2.6610496703,-2.7084293609,-2.7469704834,-2.7774727644,-2.8013054710,-2.8194269655,-2.8325316630,-2.8411724628,-2.8457918704,-2.8468011245,-2.8448086718,-2.8406285745,-2.8350634594,-2.8288797998,-2.8227815244,-2.8171723398,-2.8122692409,-2.8081443890,-2.8047605652,-2.8020128103,-2.7996644498,-2.7970025525,-2.7929566431,-2.7867662761,-2.7782027461,-2.7671727828,-2.7535195932,-2.7369486272,-2.7170492875,-2.6931922748,-2.6644974482,-2.6298251613,-2.5877551933,-2.5365915288,-2.4744496685,-2.3989845668,-2.3073322762,-2.1962528399,-2.0632937652,-1.9078262537,-1.7324744231,-1.5446953913,-1.3559533646,-1.1780693494,-1.0197532584,-0.8849816048,-0.7732678675,-0.6815782915,-0.6063465936,-0.5447967683,-0.4947465441,-0.4558163866,-0.4273576070,-0.4082726490,-0.3972108155,-0.3927668462,-0.3939374648,-0.3996968008,-0.4092360501,-0.4216462503,-0.4361957590,-0.4524766744,-0.4702600768,-0.4895810769,-0.5103847447,-0.5324601848,-0.5555518387,-0.5788781437,-0.6008854980,-0.6205969566,-0.6378309954,-0.6528570700,-0.6657854772,-0.6766269455,-0.6855496367,-0.6922395521,-0.6964978061,-0.6986225387,-0.6992073925,-0.6991366824,-0.6990716330,-0.6994178497,-0.7003477543,-0.7018818078,-0.7038874887,-0.7061441519,-0.7086060104,-0.7112855549,-0.7143035392,-0.7177705913,-0.7215952922,-0.7248177740,-0.7269684380,-0.7279907934,-0.7282027973,-0.7280837312,-0.7280595306,-0.7285101727,-0.7297127956,-0.7314122010,-0.7321141927,-0.7307092385,-0.7268683489,-0.7206026042,-0.7116615456,-0.7000355378,-0.6854536286,-0.6678141433,-0.6464909829,-0.6214520953,-0.5929967969,-0.5616410520,-0.5278019603,-0.4919950641,-0.4547454289,-0.4162090561,-0.3754768721,-0.3318636449,-0.2854391259,-0.2370200360,-0.1876200313,-0.1379282835,-0.0882704480,-0.0388885444,0.0098245314,0.0575952257,0.1043624134,0.1499894946,0.1937439687,0.2351331804,0.2730381993,0.3060707744,0.3339846136,0.3571120586,0.3757434176,0.3913850114,0.4051147660,0.4173817279,0.4284708931,0.4382117581,0.4463901261,0.4521288647,0.4551650171,0.4557584319,0.4544198242,0.4517198047,0.4481985419,0.4442559677,0.4400616421,0.4357831012,0.4315911059,0.4276681503,0.4241797508,0.4212658847,0.4189899558,0.4172617921,0.4159644981,0.4150017879,0.4143171821,0.4138921615,0.4136441769,0.4134909099,0.4133841066,0.4133039174,0.4132450687,0.4132053511,0.4131846114,0.4131839977,0.4132018933,0.4132348243,0.4132782963,0.4133265459,0.4133758346,0.4134247134,0.4134738935,0.4135225682,0.4135720252,0.4136230874,0.4136759721,0.4137305866,0.4137862645,0.4138410494,0.4138940176,0.4139455330,0.4139971339,0.4140485247,0.4141011002,0.4141556480,0.4142118116,0.4142679317,0.4143245917,0.4143821993,0.4144403209,0.4144969480,0.4145514210,0.4146048570,0.4146564952,0.4147063857,0.4147563972,0.4148072152,0.4148574872,0.4149068225,0.4149566975,0.4150079753,0.4150609025,0.4151154014,0.4151716416,0.4152310165,0.4152941018,0.4153607984,0.4154302563,0.4155000970,0.4155691691};
//float32 ver_pitch[407] = {-0.0000000000,-0.0000000000,-0.0000023860,-0.0000169390,-0.0000484306,-0.0000972015,-0.0001612824,-0.0002376561,-0.0003230994,-0.0004146676,-0.0005099286,-0.0006070300,-0.0007046697,-0.0008020141,-0.0008985958,-0.0009942144,-0.0010888516,-0.0011825967,-0.0012755927,-0.0013679994,-0.0014599703,-0.0015516381,-0.0016431040,-0.0017344436,-0.0018257104,-0.0019169398,-0.0020081530,-0.0020993612,-0.0021905683,-0.0022817741,-0.0023729761,-0.0024641707,-0.0025553574,-0.0026465469,-0.0027377515,-0.0028289822,-0.0029202486,-0.0030369422,-0.0032831210,-0.0037349338,-0.0045000546,-0.0056082896,-0.0070311540,-0.0087133834,-0.0106676667,-0.0131459453,-0.0164524968,-0.0215971692,-0.0297404101,-0.0413376592,-0.0567170943,-0.0758215337,-0.0981580601,-0.1230398655,-0.1498116771,-0.1780792822,-0.2075395680,-0.2378689695,-0.2687913826,-0.3001626393,-0.3318453017,-0.3637381919,-0.3958448080,-0.4281420096,-0.4605844550,-0.4930974200,-0.5255311550,-0.5578248083,-0.5900552944,-0.6222795782,-0.6544998576,-0.6866213675,-0.7185929766,-0.7504059341,-0.7820742819,-0.8136232549,-0.8448384650,-0.8747019288,-0.9024911634,-0.9278264932,-0.9506593850,-0.9711985695,-0.9897610135,-1.0066277622,-1.0221210855,-1.0365662934,-1.0502548729,-1.0634287255,-1.0762759655,-1.0888840966,-1.1011406856,-1.1129551037,-1.1240554822,-1.1339361958,-1.1421111760,-1.1484261352,-1.1530001536,-1.1560944674,-1.1580213248,-1.1590361645,-1.1590609998,-1.1576263018,-1.1545354551,-1.1496708091,-1.1421006084,-1.1301169937,-1.1124127377,-1.0885649444,-1.0584201516,-1.0222385456,-0.9806411759,-0.9344563416,-0.8844340004,-0.8313044129,-0.7757433614,-0.7183374466,-0.6595723907,-0.5998342186,-0.5394439646,-0.4786987508,-0.4177011338,-0.3565112424,-0.2951756210,-0.2337284250,-0.1722204752,-0.1107495047,-0.0492766868,0.0120309609,0.0723570253,0.1312998002,0.1874442044,0.2357459369,0.2754799531,0.3071775942,0.3319423258,0.3509427246,0.3652688327,0.3759218084,0.3838162880,0.3895896879,0.3937468055,0.3966904572,0.3986690916,0.3996906964,0.4000508830,0.4005296226,0.4016578080,0.4035865205,0.4061725462,0.4090237664,0.4116944613,0.4137470693,0.4147075960,0.4141508100,0.4116987106,0.4067151651,0.3988530039,0.3880829301,0.3744978873,0.3582669566,0.3394474035,0.3182415723,0.2949145690,0.2698455854,0.2433870743,0.2158347847,0.1874315064,0.1583747490,0.1288144626,0.0988435449,0.0685698899,0.0381698737,0.0079407697,-0.0220397601,-0.0518463313,-0.0816064113,-0.1114278267,-0.1413652277,-0.1714107493,-0.2015143390,-0.2316166364,-0.2616412763,-0.2914961504,-0.3210785980,-0.3502906988,-0.3790732009,-0.4073518626,-0.4350315453,-0.4620012388,-0.4881381319,-0.5132926325,-0.5372652205,-0.5599287026,-0.5810837895,-0.6005803473,-0.6183887826,-0.6344836100,-0.6488380138,-0.6614651217,-0.6724001524,-0.6817052057,-0.6894608185,-0.6957566243,-0.7006444307,-0.7040194821,-0.7057565496,-0.7058608452,-0.7045617858,-0.7022507060,-0.6993663017,-0.6962983734,-0.6933416884,-0.6906863113,-0.6884284789,-0.6866150103,-0.6853164386,-0.6845469632,-0.6843360857,-0.6848038006,-0.6875320086,-0.6970534533,-0.7162228528,-0.7453412244,-0.7831926372,-0.8278480785,-0.8771696115,-0.9290993800,-0.9817717025,-1.0335074911,-1.0828655836,-1.1285483227,-1.1692864542,-1.2036523693,-1.2301536844,-1.2473409267,-1.2540406295,-1.2498072112,-1.2351903183,-1.2117050175,-1.1814553254,-1.1465312805,-1.1085988029,-1.0688411673,-1.0282173726,-0.9874988559,-0.9479300354,-0.9103487768,-0.8752009456,-0.8426700342,-0.8127174805,-0.7854874561,-0.7608996559,-0.7388726183,-0.7190067699,-0.7007831120,-0.6839890811,-0.6684916627,-0.6544289886,-0.6417522002,-0.6302145601,-0.6195710760,-0.6094656593,-0.5990584843,-0.5875643997,-0.5745170020,-0.5598408961,-0.5437387892,-0.5266002275,-0.5088123869,-0.4907434108,-0.4725335123,-0.4542490361,-0.4359692981,-0.4176839563,-0.3992456482,-0.3806098062,-0.3615955278,-0.3417252653,-0.3206281083,-0.2980297409,-0.2739147829,-0.2483026924,-0.2213253260,-0.1930599784,-0.1635914412,-0.1327832266,-0.1006974151,-0.0674547579,-0.0331803267,0.0019474225,0.0375096066,0.0729858363,0.1080568480,0.1421906287,0.1744941557,0.2044595973,0.2321654819,0.2579130235,0.2818268436,0.3041127077,0.3247228424,0.3437447202,0.3611136547,0.3769992174,0.3917221963,0.4056623118,0.4192464599,0.4328066314,0.4465451256,0.4603861969,0.4737445893,0.4858813389,0.4964670191,0.5054219577,0.5128903529,0.5191145630,0.5242946439,0.5285896816,0.5318988317,0.5341730452,0.5355186228,0.5360957569,0.5360063067,0.5354150077,0.5345742200,0.5336664569,0.5328009917,0.5319631874,0.5306226693,0.5275333411,0.5219580879,0.5137336408,0.5031102529,0.4905415569,0.4763675080,0.4603384043,0.4422979956,0.4223223754,0.4006667324,0.3775998249,0.3534165742,0.3281973075,0.3015485272,0.2731384261,0.2427375979,0.2104446257,0.1766291610,0.1420703881,0.1078777004,0.0750995707,0.0441830776,0.0153070997,-0.0106037185,-0.0316259441,-0.0471194248,-0.0573937617,-0.0633509551,-0.0660461810,-0.0664543864,-0.0653272359,-0.0633010407,-0.0609424168,-0.0586300932,-0.0565804696,-0.0548702538,-0.0534254045,-0.0521727038,-0.0510803326,-0.0501787320,-0.0494121067,-0.0488112451,-0.0483855573,-0.0481185099,-0.0479827179,-0.0479236696,-0.0478139657,-0.0475793450,-0.0472186565,-0.0468104949,-0.0463511645,-0.0459200585,-0.0455682484,-0.0452893969,-0.0450094245,-0.0447646917,-0.0445817673,-0.0444430284,-0.0442484188,-0.0439645446,-0.0436482422,-0.0432585462,-0.0427951848,-0.0423546159,-0.0419729306,-0.0415760816,-0.0411400416,-0.0407437227,-0.0404346004,-0.0402265162,-0.0401146969,-0.0401077500,-0.0402826224,-0.0406676705,-0.0412503863,-0.0419757166,-0.0427097258,-0.0433914237};
//float32 ver_roll_rate[407] = {0.0000000000,0.0000000000,0.0002234395,0.0007197469,0.0012779750,0.0018032446,0.0022438879,0.0025823013,0.0028211565,0.0029740614,0.0030590292,0.0030943441,0.0030962789,0.0030780577,0.0030496341,0.0030179571,0.0029874877,0.0029607382,0.0029388081,0.0029218537,0.0029094641,0.0029009317,0.0028954248,0.0028921398,0.0028903792,0.0028895829,0.0028893310,0.0028893303,0.0028893916,0.0028894065,0.0028893247,0.0028891341,0.0028888543,0.0028885457,0.0028882717,0.0028880829,0.0028880125,0.0028883071,0.0028900644,0.0028944288,0.0029031410,0.0029170872,0.0029360912,0.0029590925,0.0029858301,0.0030205019,0.0030691454,0.0031527563,0.0032968144,0.0035107488,0.0037954108,0.0041313505,0.0044822839,0.0048069961,0.0050704418,0.0052523476,0.0053445259,0.0053476579,0.0052701654,0.0051266962,0.0049338948,0.0047087732,0.0044676888,0.0042241950,0.0039917016,0.0037910544,0.0036398711,0.0035496044,0.0035255062,0.0035667941,0.0036676985,0.0038182874,0.0040040902,0.0042003374,0.0043836082,0.0045361273,0.0033632705,-0.0000928736,-0.0053135382,-0.0104456396,-0.0138964673,-0.0155943604,-0.0157296972,-0.0112455597,-0.0017891933,0.0090344871,0.0194768381,0.0251060769,0.0247620038,0.0216553322,0.0174274292,0.0131561081,0.0099794554,0.0083940794,0.0076874938,0.0072673239,0.0069449330,0.0066116392,0.0062326565,0.0058112831,0.0032109176,-0.0010747638,-0.0045486504,-0.0063601156,-0.0084295667,-0.0145418452,-0.0244103441,-0.0382975193,-0.0566275608,-0.0684851871,-0.0710767546,-0.0698495148,-0.0649873637,-0.0543242809,-0.0382189812,-0.0208509990,-0.0077985261,-0.0010549113,0.0021363130,0.0032225207,0.0032607769,0.0029135564,0.0025637949,0.0023990543,0.0024805739,0.0027947949,0.0032922422,0.0039078276,0.0045679065,0.0052127065,0.0058604299,0.0045307844,0.0011254380,-0.0021837184,-0.0045915274,-0.0058350869,-0.0059970633,-0.0053334952,-0.0041244171,-0.0026482592,-0.0011377768,-0.0031914347,-0.0141730632,-0.0307918937,-0.0763848042,-0.1759261518,-0.3136608348,-0.4881004273,-0.6904362732,-0.9078204077,-1.1271653247,-1.3683554004,-1.6522040293,-1.9387026318,-2.1917743279,-2.3975540907,-2.5536509796,-2.6644230043,-2.7374850932,-2.7813876872,-2.8045148081,-2.8138952366,-2.8149964430,-2.8115570520,-2.8094631689,-2.8109793082,-2.8102774775,-2.8055525863,-2.8028686594,-2.8043277814,-2.8075213712,-2.8075842638,-2.8016130428,-2.7916356793,-2.7788560805,-2.7646344339,-2.7502634936,-2.7368402723,-2.7251692218,-2.7191193347,-2.7194619568,-2.7227926980,-2.7273901588,-2.7287192549,-2.7253968848,-2.7202563451,-2.7146310095,-2.7093826474,-2.7049539050,-2.7015136614,-2.6853710486,-2.6363559618,-2.4708732329,-2.1904913349,-1.7833762624,-1.2696015515,-0.7754811800,-0.3539682540,-0.0103708452,0.2727873314,0.5058137472,0.6972926904,0.8543814051,0.9767256743,1.0553354105,1.0868519718,1.0738477108,1.0167100060,0.9265304905,0.8189361122,0.7039574010,0.5907100579,0.4856771583,0.3928071672,0.3276046658,0.3172785908,0.3635420347,0.4385965789,0.5174923318,0.5503359990,0.5165176684,0.4466828841,0.3624800744,0.2780740764,0.2018369311,0.1379065955,0.0875175924,0.0466619228,0.0130624783,-0.0120115179,-0.0257301373,-0.0283692990,-0.0242345548,-0.0157528164,-0.0046104120,0.0046591113,0.0105866753,0.0022853299,-0.0360250368,-0.1063237871,-0.1960089943,-0.2863158694,-0.3825962594,-0.4835362180,-0.6209973018,-0.8242776186,-1.0626209115,-1.3030396975,-1.5200804041,-1.7099021800,-1.8716638143,-2.0052428416,-2.1102362602,-2.1802907988,-2.2208094530,-2.2411809008,-2.2521934450,-2.2606441925,-2.2639627532,-2.2623178804,-2.2456661101,-2.1746146581,-2.0457726847,-1.8985229890,-1.7596312968,-1.6289742362,-1.4914805800,-1.3559898019,-1.2136881555,-1.0538439970,-0.9032965422,-0.7805236396,-0.7017375319,-0.6712903682,-0.6719442338,-0.6888126808,-0.7150257647,-0.7409993466,-0.7520044247,-0.7467915355,-0.7293025825,-0.7034990279,-0.6728843592,-0.6406781380,-0.5708251932,-0.4519432984,-0.3164498469,-0.1848437032,-0.0746285558,0.0113076552,0.0733118197,0.1113295642,0.1323370882,0.1896622190,0.3057760899,0.4405320888,0.5665685176,0.6918007589,0.8165806609,0.9431262667,1.0709786338,1.2152346404,1.3778657209,1.5325330331,1.6688850418,1.7870315201,1.8881317395,1.9652652354,2.0189865512,2.0907134872,2.2037096380,2.3285534959,2.4291032876,2.4816942885,2.4940896627,2.4868057380,2.4717364946,2.4429227054,2.3979502085,2.3477925777,2.3010491572,2.2304420176,2.1304480629,1.9955734203,1.7849622133,1.5272073007,1.2807933093,1.0436956800,0.8649774506,0.7768949771,0.7281443356,0.6979513854,0.6634167248,0.6150366121,0.5340206473,0.4182979873,0.3026239313,0.2015530811,0.1217654346,0.0641849263,0.0265633729,0.0017496517,-0.0150958271,-0.0245246046,-0.0282512164,-0.0279296879,-0.0250427706,-0.0207991582,-0.0195331505,-0.0226173339,-0.0271748519,-0.0319185024,-0.0326705641,-0.0283728361,-0.0222102326,-0.0157221665,-0.0098462738,-0.0050151714,-0.0013427582,0.0012460006,0.0029214486,0.0038864941,0.0043376179,0.0044446329,0.0043426620,0.0041317288,0.0038809324,0.0036341371,0.0034160176,0.0032374496,0.0031004068,0.0030015063,0.0029346912,0.0028930528,0.0028699441,0.0028595970,0.0028575695,0.0028606811,0.0028664377,0.0028731724,0.0028794938,0.0028845291,0.0028880215,0.0028903071,0.0028914311,0.0028915520,0.0028910004,0.0028903178,0.0028897223,0.0028891290,0.0028886014,0.0028880867,0.0028874712,0.0028867696,0.0028860319,0.0028852701,0.0028845545,0.0028839801,0.0028836185,0.0028834973,0.0028836367,0.0028841525,0.0028850762,0.0028863432,0.0028877794,0.0028890478,0.0028899607};
//float32 ver_pitch_rate[407] = {0.0000000000,0.0000000000,-0.0003523807,-0.0011350743,-0.0020153308,-0.0028434226,-0.0035378250,-0.0040707863,-0.0044465135,-0.0046864707,-0.0048191110,-0.0048734045,-0.0048751590,-0.0048453465,-0.0047997385,-0.0047493061,-0.0047009943,-0.0046586528,-0.0046239163,-0.0045969579,-0.0045770839,-0.0045631777,-0.0045540121,-0.0045484163,-0.0045453702,-0.0045440437,-0.0045437998,-0.0045441757,-0.0045448538,-0.0045456305,-0.0045463863,-0.0045470598,-0.0045476226,-0.0045480507,-0.0045483325,-0.0045484686,-0.0045484690,-0.0082970682,-0.0166234095,-0.0297365675,-0.0468731361,-0.0636263136,-0.0781083034,-0.0894967388,-0.1089708541,-0.1393653561,-0.2005871152,-0.3255411045,-0.4905789227,-0.6727935208,-0.8646039130,-1.0412211595,-1.1863856400,-1.2956614595,-1.3785435194,-1.4457515355,-1.4973766767,-1.5329301757,-1.5583682125,-1.5774655107,-1.5896799799,-1.5999361112,-1.6104165898,-1.6188874148,-1.6249183851,-1.6247614201,-1.6179558969,-1.6123982203,-1.6110388577,-1.6115876569,-1.6092510619,-1.6024967349,-1.5945390218,-1.5868164608,-1.5801614800,-1.5749010568,-1.5351913741,-1.4456398533,-1.3298441329,-1.2033218943,-1.0818156874,-0.9750244749,-0.8832563102,-0.8062117251,-0.7458630763,-0.7011141118,-0.6697780858,-0.6491846738,-0.6366995046,-0.6224990889,-0.6030060446,-0.5763203811,-0.5292053921,-0.4542659874,-0.3622471630,-0.2705693296,-0.1893313926,-0.1230313240,-0.0725459726,-0.0291061097,0.0317412908,0.1127380158,0.1951515120,0.2984978021,0.4746106566,0.7353901737,1.0367106683,1.3500219546,1.6620386276,1.9509928883,2.2014177152,2.4114960629,2.5850264314,2.7228873099,2.8290174990,2.9082791767,2.9657806368,3.0063320659,3.0304028051,3.0444116751,3.0554975533,3.0638148626,3.0701684377,3.0750549763,3.0750991248,3.0734894352,3.0747902050,3.0472074795,2.9834886207,2.9114585775,2.6393275450,2.1972996464,1.7794561274,1.4019532147,1.0854090049,0.8254382087,0.6175627287,0.4589868890,0.3396139712,0.2479542419,0.1793921264,0.1289325592,0.0835354856,0.0417844672,0.0318885279,0.0615787484,0.1110608269,0.1653284292,0.2116449593,0.2436125047,0.2629535294,0.2693443798,0.2631011170,0.2512339777,0.2225824719,0.1723945365,0.1181166290,0.0664848687,0.0208935926,-0.0258130433,-0.0759818798,-0.1241454620,-0.1646266016,-0.1924893410,-0.2105622730,-0.2209673277,-0.2259246678,-0.2274708662,-0.2309030091,-0.2380002729,-0.2458321224,-0.2194765363,-0.1472436528,-0.0563447861,0.0360711397,0.1168986738,0.1808616213,0.2270802954,0.2572723575,0.2781711742,0.2940281480,0.3044426165,0.3103970683,0.3129284278,0.3166944423,0.3233452686,0.3300367500,0.3357267880,0.3399545410,0.3426995816,0.3402890368,0.3352391886,0.3266556640,0.3080943410,0.2838612387,0.2525747026,0.2117252153,0.1632741321,0.1124171708,0.0624973989,0.0137980738,-0.0332749298,-0.0782873077,-0.1270620253,-0.1850442175,-0.2480430903,-0.3050107383,-0.3444399039,-0.3606176554,-0.3540665187,-0.3294096320,-0.2926605241,-0.2496996379,-0.2054322429,-0.1597294140,-0.1141381669,-0.0696734708,-0.0273178391,0.0352274067,0.2807147556,0.7597169425,1.3230418615,1.8620144390,2.3218531782,2.6813994900,2.9409276876,3.1125210113,3.2096639777,3.2496036677,3.2523105500,3.2367092193,3.2111404108,3.1789108235,3.1504861893,3.1299889214,3.1154722471,3.1057162709,3.0957658684,3.0715495650,3.0265778183,2.9682422565,2.9066037440,2.8387601953,2.7665870468,2.6657022494,2.5270606428,2.3786260486,2.2305410871,2.0956213918,1.9672840137,1.8445228812,1.7322229240,1.6354581684,1.5682535003,1.5169652674,1.4711255772,1.4196513951,1.3585761212,1.3043864612,1.2623875232,1.2306535547,1.2296830769,1.2738579095,1.3496157592,1.4359769756,1.5147969000,1.5742694016,1.6111587505,1.6264532046,1.6311526744,1.6385564574,1.6478818390,1.6601280269,1.6867593618,1.7268091221,1.7759429326,1.8530893962,1.9610850523,2.0875935473,2.2168294835,2.3374042425,2.4437110067,2.5347551447,2.6111878521,2.6821056189,2.7454502101,2.7887326912,2.8141145495,2.8246806691,2.8132981166,2.7664306762,2.6946925684,2.6044699550,2.4654396138,2.2734581570,2.0684015014,1.8831968363,1.7135625592,1.5593422533,1.4133035068,1.2727288576,1.1376045572,1.0081545479,0.9012653497,0.8208907957,0.7698752542,0.7452374138,0.7385330156,0.7395074835,0.7234273852,0.6672745590,0.5837508113,0.4941514658,0.4094663426,0.3387440536,0.2809478631,0.2347493175,0.1901584818,0.1395345130,0.0905172451,0.0490613103,0.0134272656,-0.0172869917,-0.0370211834,-0.0450539402,-0.0458161658,-0.0421822612,-0.0476026200,-0.1073791147,-0.2285868736,-0.3734520390,-0.5156944732,-0.6380665152,-0.7339863332,-0.8296778893,-0.9417152153,-1.0550241351,-1.1568482530,-1.2424986209,-1.3110704708,-1.3644719843,-1.4282148122,-1.5144726446,-1.6166783545,-1.7228696628,-1.8161944765,-1.8789831989,-1.8884600673,-1.8340679639,-1.7377688538,-1.6320066113,-1.5070107979,-1.2902018127,-0.9929429035,-0.6933361558,-0.4299425329,-0.2233659123,-0.0739433298,0.0282859014,0.0938731642,0.1259572667,0.1317074585,0.1217796875,0.1036077859,0.0862243602,0.0738418928,0.0648441909,0.0547591328,0.0458056671,0.0380518917,0.0282933772,0.0190259114,0.0110417441,0.0047103261,0.0037875584,0.0089172800,0.0162863172,0.0202618459,0.0226278926,0.0234844864,0.0199255666,0.0148672929,0.0132966977,0.0130772950,0.0101989138,0.0067022312,0.0069959654,0.0123356138,0.0155999584,0.0181318854,0.0233621750,0.0251896329,0.0219857689,0.0205859985,0.0232833379,0.0237106579,0.0198914468,0.0146382572,0.0092022333,0.0043517659,-0.0033189100,-0.0145610807,-0.0259139460,-0.0359409597,-0.0402065622,-0.0377588334,-0.0360786010};
//float32 ver_yaw_rate[407] = {0.0000000000,0.0000000000,0.0002846600,0.0010623340,0.0022271938,0.0034168016,0.0043494206,0.0053175712,0.0065404417,0.0079424806,0.0091914388,0.0100350863,0.0105256392,0.0107120880,0.0106427654,0.0106325902,0.0109217431,0.0114506640,0.0121670837,0.0130228234,0.0139744018,0.0147148994,0.0150145143,0.0149448707,0.0145713747,0.0139558842,0.0131556527,0.0122226332,0.0112030507,0.0101372056,0.0090594713,0.0079984501,0.0072457159,0.0070157662,0.0072221259,0.0077853157,0.0086304253,0.0096883403,0.0108965456,0.0121996054,0.0135493658,0.0149049258,0.0162324199,0.0175046521,0.0187006187,0.0198049530,0.0208073200,0.0217017885,0.0224861985,0.0231615440,0.0234629229,0.0231994106,0.0224778961,0.0213951478,0.0200403516,0.0184940540,0.0168275737,0.0151027668,0.0133720879,0.0116788899,0.0100579135,0.0085359159,0.0071323984,0.0058603946,0.0049957484,0.0047375083,0.0049843861,0.0056429981,0.0066256381,0.0078516384,0.0092482498,0.0107511502,0.0120361733,0.0128596595,0.0132825066,0.0133622337,0.0132243866,0.0129738853,0.0126377055,0.0121756920,0.0115600691,0.0111004051,0.0110293261,0.0112788927,0.0117879807,0.0124993906,0.0133607133,0.0143248651,0.0153503830,0.0164015162,0.0174481475,0.0184655792,0.0194038761,0.0202341971,0.0209837041,0.0216676152,0.0222813279,0.0225545733,0.0222911435,0.0215928548,0.0206677131,0.0195746426,0.0182642851,0.0170425738,0.0160049061,0.0149143497,0.0137586790,0.0125000670,0.0110782825,0.0094547939,0.0075308870,0.0056009625,0.0039583788,0.0025873388,0.0014713068,0.0005902520,-0.0000781635,-0.0008260557,-0.0018740479,-0.0031462240,-0.0045770374,-0.0061079253,-0.0076876872,-0.0092725496,-0.0108260197,-0.0123185689,-0.0137271856,-0.0150348343,-0.0162298522,-0.0173053106,-0.0182307431,-0.0206250416,-0.0257012792,-0.0342075394,-0.0466011159,-0.0634953555,-0.0851885183,-0.1129676591,-0.1486350094,-0.1919980513,-0.2416740370,-0.2969713011,-0.3590649175,-0.4294702091,-0.5070730618,-0.5894015341,-0.6753040418,-0.7628101405,-0.8497152531,-0.9346129571,-1.0161251573,-1.0926881289,-1.1634174613,-1.2279546771,-1.2856423523,-1.3361275151,-1.3795421289,-1.4164297532,-1.4473803754,-1.4725142116,-1.4920920002,-1.5067087755,-1.5172353692,-1.5242046665,-1.5279110098,-1.5289221187,-1.5277645978,-1.5249212611,-1.5210910679,-1.5168516248,-1.5124377092,-1.5077805292,-1.5028420078,-1.4981035789,-1.4939320719,-1.4903656810,-1.4874199950,-1.4848243715,-1.4823615437,-1.4800925224,-1.4780574686,-1.4762817918,-1.4747782325,-1.4735489336,-1.4725874068,-1.4718803591,-1.4714093613,-1.4711523402,-1.4710848898,-1.4711814002,-1.4714160067,-1.4717633692,-1.4721992904,-1.4727011887,-1.4732484373,-1.4738225879,-1.4717641839,-1.4651246697,-1.4461697538,-1.4089616304,-1.3566687758,-1.2919096675,-1.2168482557,-1.1336113436,-1.0444190200,-0.9511224119,-0.8555844889,-0.7601803118,-0.6668283426,-0.5769507122,-0.4919027752,-0.4126939023,-0.3400245762,-0.2743217427,-0.2157751867,-0.1643737254,-0.1199402794,-0.0821650867,-0.0506365074,-0.0253944268,-0.0057808961,0.0093344863,0.0205089461,0.0280282268,0.0322275961,0.0337071974,0.0330178941,0.0295848117,0.0230670051,0.0142543730,0.0032999787,-0.0122547655,-0.0352328883,-0.0657215356,-0.1021233194,-0.1427621192,-0.1861372860,-0.2341333870,-0.2892011970,-0.3506258441,-0.4168813509,-0.4858987661,-0.5558116998,-0.6250427914,-0.6934586840,-0.7618671098,-0.8295117869,-0.8946243177,-0.9558887489,-1.0125586072,-1.0641279739,-1.1103053510,-1.1501902691,-1.1769051458,-1.1857452955,-1.1792156869,-1.1601908859,-1.1313547804,-1.0951691695,-1.0538484755,-1.0088170710,-0.9578437079,-0.8995277050,-0.8367458851,-0.7723116101,-0.7096668522,-0.6536174696,-0.6063376996,-0.5692942576,-0.5454469106,-0.5360887137,-0.5398951492,-0.5552678919,-0.5808141685,-0.6143681908,-0.6537024390,-0.6960428653,-0.7374935931,-0.7728036924,-0.7994650620,-0.8156793948,-0.8203792566,-0.8138589742,-0.7967281582,-0.7679102180,-0.7268733189,-0.6763371709,-0.6189096866,-0.5571693591,-0.4933515750,-0.4293540378,-0.3667469197,-0.3067899638,-0.2504547644,-0.1984506842,-0.1512530546,-0.1091324894,-0.0721843285,-0.0403574060,-0.0134814972,0.0087070404,0.0265417886,0.0404051789,0.0507099195,0.0578829731,0.0623520462,0.0645344928,0.0648284868,0.0636062854,0.0612093830,0.0579453389,0.0540860617,0.0498673297,0.0454893364,0.0411180625,0.0368872865,0.0329010675,0.0289680870,0.0249450659,0.0209641729,0.0171285950,0.0135184307,0.0101927801,0.0074605099,0.0055422035,0.0043481911,0.0037883340,0.0037712771,0.0042071988,0.0050099692,0.0060988122,0.0073995179,0.0085767957,0.0093751643,0.0098434615,0.0100272875,0.0099716659,0.0097199010,0.0093127342,0.0087877219,0.0084472737,0.0085179779,0.0089261608,0.0096053892,0.0104936575,0.0115343015,0.0124080968,0.0128739822,0.0129934774,0.0128230978,0.0124170395,0.0118261037,0.0110969524,0.0102716102,0.0093871812,0.0084757495,0.0075644315,0.0066755512,0.0058269115,0.0050321363,0.0043010622,0.0036401590,0.0030529638,0.0025405137,0.0021017673,0.0017340037,0.0014331945,0.0011943427,0.0010117858,0.0008794624,0.0007911397,0.0007406060,0.0007218267,0.0007290682,0.0004885301,-0.0002011642,-0.0012445847,-0.0022884937,-0.0030593291,-0.0035882675,-0.0039040631,-0.0040360120,-0.0040130267,-0.0038629342,-0.0036119311,-0.0032841811,-0.0029015413,-0.0024833989,-0.0020466042,-0.0016054828,-0.0011719129,-0.0007554536,-0.0003635116,-0.0000015340,0.0003267811,0.0006192665,0.0008750665,0.0010944231,0.0012784715,0.0014290496,0.0015485227,0.0016396270,0.0017053314,0.0017487174,0.0017728795,0.0017808413,0.0017754905};
//float32 ver_ail[407] = {0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0150156298,0.0150156298,0.0150156298,0.0150156298,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,-0.0131790235,-0.0131790235,-0.0131790235,-0.0131790235,0.0150156298,0.0009183032,0.0009183032,0.0009183032,0.0150156298,0.0150156298,0.0150156298,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0150156298,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,-0.0131790235,-0.0413736768,-0.0836656567,-0.3233202096,-0.4360988228,-0.4360988228,-0.5347801093,-0.5347801093,-0.6052667425,-0.6052667425,-0.7884769212,-0.8730608811,-0.8730608811,-0.8730608811,-0.8730608811,-0.8730608811,-0.8730608811,-0.8730608811,-0.8730608811,-0.8730608811,-0.8730608811,-0.8730608811,-0.8730608811,-0.8871582077,-0.8871582077,-0.8730608811,-0.8730608811,-0.8871582077,-0.8871582077,-0.8871582077,-0.8730608811,-0.8730608811,-0.8730608811,-0.8730608811,-0.8730608811,-0.8730608811,-0.8730608811,-0.8730608811,-0.8871582077,-0.8871582077,-0.8871582077,-0.8871582077,-0.8730608811,-0.8730608811,-0.8730608811,-0.8730608811,-0.8730608811,-0.8730608811,-0.8730608811,-0.8166715745,-0.7461849413,-0.3656121896,-0.3656121896,0.0855022630,0.0855022630,0.0855022630,0.0855022630,-0.1259576366,-0.1259576366,-0.1259576366,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0150156298,0.0150156298,0.0150156298,0.0150156298,0.0150156298,0.0150156298,0.0714049364,0.1700862229,0.2124332705,0.2124332705,0.2124332705,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,-0.0131790235,-0.0131790235,-0.0131790235,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,-0.0131790235,-0.0131790235,-0.0695683301,-0.1541522899,-0.2246389231,-0.2246389231,-0.2246389231,-0.2951255564,-0.2951255564,-0.5911694159,-0.6616560491,-0.6897956347,-0.7038929613,-0.7038929613,-0.7461849413,-0.7461849413,-0.7743795946,-0.7602822679,-0.7461849413,-0.7461849413,-0.7461849413,-0.7602822679,-0.7602822679,-0.7461849413,-0.7461849413,-0.6897956347,-0.5206827826,-0.4924881294,-0.4924881294,-0.4924881294,-0.4360988228,-0.3656121896,-0.3656121896,-0.2528335764,-0.2105415965,-0.2246389231,-0.2246389231,-0.2810282297,-0.2951255564,-0.2951255564,-0.2951255564,-0.3092228830,-0.2951255564,-0.2669309031,-0.2669309031,-0.2528335764,-0.2528335764,-0.2387362498,-0.2387362498,-0.0554710034,-0.0413736768,-0.0131790235,-0.0131790235,-0.0131790235,0.0009183032,0.0009183032,0.0009183032,0.0150156298,0.2123782028,0.2553860626,0.2553860626,0.2553860626,0.3413467145,0.3413467145,0.4272522988,0.4272522988,0.5704282725,0.5704282725,0.5991185349,0.5991185349,0.6420713270,0.6420713270,0.6420713270,0.6420713270,0.7995648981,0.8425176902,0.8425176902,0.7852473008,0.7422945086,0.7422945086,0.7566121060,0.7566121060,0.7136593139,0.6993417165,0.6993417165,0.6993417165,0.5704282725,0.5704282725,0.4272522988,0.2410684652,0.2410684652,0.2410684652,0.1277942430,0.3126564521,0.2983388547,0.2410684652,0.2410684652,0.1559888962,0.1559888962,0.0150156298,0.0150156298,0.0150156298,0.0150156298,0.0150156298,0.0150156298,0.0150156298,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,-0.0131790235,-0.0131790235,-0.0131790235,-0.0131790235,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032,0.0009183032};
//float32 ver_ele[407] = {-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0168549377,-0.0168549377,-0.0322616394,-0.0322616394,-0.0322616394,-0.0322616394,-0.0322616394,-0.0784817444,-0.0784817444,-0.2017353579,-0.3269749916,-0.3269749916,-0.4052723310,-0.4365671938,-0.4365671938,-0.4365671938,-0.4365671938,-0.4678620566,-0.4835094880,-0.4835094880,-0.4835094880,-0.4991569194,-0.4991569194,-0.4991569194,-0.5148043508,-0.5148043508,-0.5148043508,-0.5148043508,-0.4991569194,-0.4991569194,-0.5148043508,-0.5148043508,-0.5148043508,-0.4991569194,-0.4991569194,-0.4991569194,-0.4991569194,-0.4991569194,-0.4991569194,-0.2800326974,-0.2487378346,-0.2174429717,-0.2174429717,-0.2174429717,-0.2174429717,-0.2017955403,-0.2017955403,-0.2017955403,-0.2017955403,-0.2017955403,-0.2017353579,-0.2017353579,-0.1709219545,-0.1709219545,-0.1401085512,-0.0630750427,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,0.0293651674,0.1372120792,0.1372120792,0.1372120792,0.3220924995,0.6918533400,0.8920802795,0.8920802795,0.9845204896,0.9845204896,0.9845204896,0.9691137879,0.9845204896,0.9845204896,0.9845204896,0.9845204896,0.9845204896,0.9845204896,0.9845204896,0.9691137879,0.9845204896,0.9845204896,0.9845204896,0.9845204896,0.9845204896,0.9691137879,0.9845204896,0.9845204896,0.8612668761,0.8612668761,0.8612668761,-0.6243965530,-0.0476683410,-0.0476683410,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,0.0139584657,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0476683410,-0.0784817444,0.0139584657,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0168549377,-0.0322616394,-0.0322616394,-0.1092951478,-0.1247018495,-0.1247018495,-0.1401085512,-0.1401085512,-0.1863286562,-0.1863286562,-0.2017955403,-0.1863286562,-0.1863286562,-0.1863286562,-0.1863286562,-0.1863286562,-0.1863286562,-0.2017353579,-0.2017353579,-0.2017353579,-0.0630750427,-0.0476683410,-0.0322616394,-0.0322616394,-0.0322616394,-0.0322616394,-0.0322616394,-0.0322616394,-0.0168549377,-0.0168549377,-0.0168549377,-0.0168549377,-0.0168549377,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0168549377,-0.0014482360,-0.0168549377,-0.0168549377,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,0.0139584657,-0.0014482360,0.0139584657,0.0139584657,0.0139584657,0.0139584657,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,0.0139584657,0.0139584657,0.0293651674,0.0293651674,0.1372120792,0.8458601744,0.9845204896,0.9845204896,0.9845204896,0.9845204896,0.9845204896,0.9845204896,0.9845204896,0.9691137879,0.9691137879,0.9691137879,0.9845204896,0.9691137879,0.9691137879,0.9845204896,0.9845204896,0.9845204896,0.9845204896,0.9691137879,0.8920802795,0.8458601744,0.8458601744,0.8458601744,0.7996400694,0.7996400694,0.5531930248,0.5531930248,0.5377863231,0.5069729197,0.5069729197,0.4299394113,0.4299394113,0.3837193062,0.3991260079,0.4145327096,0.3683126045,0.3683126045,0.3066857978,0.3066857978,0.3220924995,0.3220924995,0.3220924995,0.4299394113,0.4761595163,0.5069729197,0.5069729197,0.5069729197,0.4915662180,0.4915662180,0.4761595163,0.5069729197,0.5223796214,0.5223796214,0.5377863231,0.5840064282,0.5840064282,0.6148198315,0.7072600417,0.7380734450,0.7842333677,0.7842333677,0.8150467710,0.8150467710,0.8458601744,0.8458601744,0.8920802795,0.8766735778,0.8766735778,0.8766735778,0.8766735778,0.8304534727,0.7688868484,0.7688868484,0.7072600417,0.5531930248,0.4915662180,0.4915662180,0.4915662180,0.4145327096,0.4145327096,0.3220924995,0.3220924995,0.2450589910,0.2450589910,0.2450589910,0.2450589910,0.2604656927,0.2604656927,0.2604656927,0.2450589910,0.1680254826,0.0909919741,0.0909919741,0.0755852725,0.0755852725,0.0755852725,0.0601785708,0.0601785708,0.0139584657,-0.0014482360,-0.0014482360,-0.0014482360,-0.0168549377,-0.0168549377,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0476683410,-0.2174429717,-0.2487378346,-0.2643852660,-0.2643852660,-0.2643852660,-0.2643852660,-0.3739774682,-0.4052723310,-0.4209197624,-0.4209197624,-0.4365671938,-0.4365671938,-0.4522146252,-0.5460992136,-0.5773940764,-0.6243965530,-0.6243965530,-0.6243965530,-0.5773940764,-0.4991569194,-0.4209197624,-0.4209197624,-0.4209197624,-0.3113275602,-0.0014482360,-0.0014482360,0.0139584657,-0.0014482360,-0.0014482360,-0.0014482360,0.0139584657,0.0139584657,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,0.0139584657,0.0139584657,0.0139584657,-0.0014482360,0.0139584657,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,0.0139584657,0.0139584657,0.0139584657,-0.0014482360,0.0139584657,-0.0014482360,-0.0014482360,-0.0014482360,0.0139584657,-0.0014482360,-0.0014482360,-0.0014482360,0.0139584657,0.0139584657,-0.0014482360,0.0139584657,0.0139584657,-0.0014482360,-0.0014482360,0.0139584657,0.0139584657,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0014482360,-0.0168549377,-0.0168549377,-0.0168549377,-0.0168549377,-0.0014482360,-0.0014482360,-0.0168549377,-0.0014482360,-0.0014482360};
//float32 ver_rud[407] = {0.0155764122,0.0155764122,0.0155764122,0.0008863956,0.0008863956,0.0155764122,0.0155764122,0.0155764122,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0155764122,0.0008863956,0.0008863956,0.0008863956,-0.0138036210,-0.0138036210,-0.0138036210,-0.0138036210,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,-0.0138036210,-0.0138036210,-0.0138036210,-0.0138036210,-0.0138036210,-0.0138036210,-0.0138036210,-0.0138036210,-0.0138036210,-0.0138036210,-0.0138036210,-0.0138036210,-0.0138036210,-0.0138036210,-0.1020002206,-0.1020002206,-0.1754503036,-0.1754503036,-0.2622909017,-0.2622909017,-0.4069310651,-0.4647871305,-0.5226431958,-0.5226431958,-0.6094272939,-0.7251394246,-0.8987076206,-0.9420996697,-0.9565636860,-0.9565636860,-0.9420996697,-0.9420996697,-0.9565636860,-0.9565636860,-0.9420996697,-0.9565636860,-0.9565636860,-0.9420996697,-0.9420996697,-0.9420996697,-0.9565636860,-0.9565636860,-0.9420996697,-0.9420996697,-0.9420996697,-0.9565636860,-0.9420996697,-0.9420996697,-0.9420996697,-0.9420996697,-0.9420996697,-0.9565636860,-0.9565636860,-0.9565636860,-0.9420996697,-0.9420996697,-0.9565636860,-0.9565636860,-0.9565636860,-0.9565636860,-0.9420996697,-0.9420996697,-0.9420996697,-0.9420996697,-0.9420996697,-0.9420996697,-0.9420996697,-0.9420996697,-0.9420996697,-0.9420996697,-0.9420996697,-0.9420996697,-0.9420996697,-0.9420996697,-0.9420996697,-0.9420996697,-0.9420996697,-0.9420996697,-0.9420996697,-0.7974595063,-0.7974595063,0.5739100429,0.5739100429,0.5739100429,0.1771665947,0.0596464620,0.0008863956,0.0008863956,0.0302664288,0.0302664288,0.0008863956,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0008863956,0.0008863956,0.0008863956,0.0008863956,-0.0579301708,-0.0579301708,-0.0579301708,-0.0873102040,-0.2333628690,-0.3201469671,-0.3490749997,-0.3490749997,-0.3635390161,-0.3635390161,-0.5515712285,-0.6383553265,-0.6817473756,-0.6817473756,-0.6817473756,-0.6962113919,-0.6962113919,-0.8408515553,-0.8408515553,-0.8263875389,-0.8119235226,-0.8119235226,-0.8119235226,-0.8119235226,-0.8119235226,-0.7685314736,-0.4213950814,-0.3924670488,-0.3924670488,-0.3924670488,-0.3924670488,-0.3924670488,-0.3924670488,-0.3635390161,-0.1607602870,-0.1460702704,-0.1755068037,-0.1755068037,-0.2622909017,-0.3780030324,-0.3780030324,-0.4792511468,-0.5949632775,-0.6528193429,-0.6528193429,-0.6962113919,-0.7106754082,-0.7106754082,-0.7106754082,-0.6672833592,-0.5804992612,-0.4213950814,-0.4213950814,-0.2767549180,-0.2767549180,-0.1899708200,-0.1899708200,0.0008863956,0.0008863956,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0155764122,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,-0.0138036210,-0.0138036210,-0.0138036210,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0008863956,0.0155764122};
//float32 ver_thr[407] = {0.4878208816,0.4878208816,0.4878208816,0.4878208816,0.4878208816,0.4814873157,0.4814873157,0.4814873157,0.4878208816,0.4878208816,0.4814873157,0.4814873157,0.4814873157,0.4814873157,0.4878208816,0.4878208816,0.4878208816,0.4878208816,0.4878208816,0.4878208816,0.4814873157,0.4814873157,0.4814873157,0.4814873157,0.4814873157,0.4814873157,0.4814873157,0.4878208816,0.4878208816,0.4878208816,0.5005123733,0.5005123733,0.5005123733,0.5131795051,0.5131795051,0.5131795051,0.5131795051,0.5131795051,0.5131795051,0.5131795051,0.5131795051,0.5131795051,0.5131795051,0.5131795051,0.5131795051,0.5448473348,0.5448473348,0.5511078211,0.5511078211,0.5511078211,0.5511078211,0.5635800740,0.5698162004,0.5698162004,0.5698162004,0.5698162004,0.5698162004,0.5698162004,0.5760523268,0.5698162004,0.5698162004,0.5760523268,0.5698162004,0.5822884533,0.5822884533,0.6009968326,0.6009968326,0.6820664764,0.6883026029,0.6883026029,0.7070109822,0.7631361202,0.7631361202,0.7631361202,0.9626678066,0.9689039331,0.9689039331,0.9689039331,0.9689039331,0.9689039331,0.9689039331,0.9751400595,0.9689039331,0.9751400595,0.9751400595,0.9689039331,0.9689039331,0.9751400595,0.9751400595,0.9751400595,0.9751400595,0.9689039331,0.9751400595,0.9751400595,0.9751400595,0.9751400595,0.9689039331,0.9689039331,0.9689039331,0.9689039331,0.9689039331,0.9689039331,0.9689039331,0.9689039331,0.9689039331,0.9626678066,0.9626678066,0.9564316802,0.9564316802,0.9564316802,0.9564316802,0.9564316802,0.9564316802,0.9564316802,0.9564316802,0.9564316802,0.9501955537,0.9501955537,0.9252754078,0.9252754078,0.9252754078,0.9190392814,0.8816225227,0.8753863962,0.8566780169,0.8442057640,0.8442057640,0.8442057640,0.8317335111,0.7257193615,0.7257193615,0.7007748558,0.7007748558,0.6758303500,0.6758303500,0.6571219706,0.6571219706,0.6571219706,0.6571219706,0.6571219706,0.6321774649,0.6321774649,0.6321774649,0.6321774649,0.6321774649,0.6321774649,0.6321774649,0.6259413384,0.6259413384,0.6197052120,0.6259413384,0.6259413384,0.6197052120,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6197052120,0.6197052120,0.6197052120,0.6259413384,0.6197052120,0.6259413384,0.6259413384,0.6197052120,0.6197052120,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6197052120,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6197052120,0.6197052120,0.6197052120,0.6197052120,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6197052120,0.6197052120,0.6197052120,0.6197052120,0.6321774649,0.6321774649,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6197052120,0.6259413384,0.6197052120,0.6197052120,0.6197052120,0.6197052120,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6197052120,0.6197052120,0.6197052120,0.6197052120,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6197052120,0.6197052120,0.6197052120,0.6259413384,0.6197052120,0.6197052120,0.6197052120,0.6197052120,0.6197052120,0.6197052120,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6197052120,0.6197052120,0.6259413384,0.6259413384,0.6259413384,0.6259413384,0.6072329591,0.6009968326,0.6009968326,0.6009968326,0.6009968326,0.5947607062,0.5947607062,0.6009968326,0.6009968326,0.5947607062,0.5947607062,0.5947607062,0.6009968326,0.6009968326,0.6009968326,0.6009968326,0.6009968326,0.6009968326,0.6009968326,0.6072329591,0.6072329591,0.6072329591,0.6009968326,0.6009968326,0.6009968326,0.6072329591,0.6072329591,0.5947607062,0.5947607062,0.6009968326,0.6009968326,0.6009968326,0.6009968326,0.6009968326,0.6009968326,0.6009968326,0.6009968326,0.6009968326,0.6009968326,0.6009968326,0.6820664764,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6883026029,0.6820664764,0.6758303500,0.6758303500,0.6758303500,0.6820664764,0.6820664764,0.6695942235,0.6695942235,0.6695942235,0.6571219706,0.6571219706,0.6571219706,0.6571219706,0.6571219706,0.6384135913,0.6072329591,0.6072329591,0.6072329591,0.5947607062,0.5947607062,0.5947607062,0.5947607062,0.5947607062,0.5947607062,0.5947607062,0.5947607062,0.5947607062,0.5947607062,0.5947607062,0.5822884533,0.5635800740,0.5448473348,0.5321802029,0.5321802029,0.5321802029,0.5321802029,0.5321802029,0.5321802029,0.5131795051,0.4814873157,0.4814873157};

void control_step(){
//	float32 roll = ver_roll[i];
//	float32 pitch = ver_pitch[i];
//	float32 roll_rate = ver_roll_rate[i];
//	float32 pitch_rate = ver_pitch_rate[i];
//	float32 yaw_rate = ver_yaw_rate[i];
//	roll_ref = ver_ail[i];
//	pitch_ref = ver_ele[i];
//	yaw_rate_ref = ver_rud[i];
//	thr_ref = ver_thr[i];
	if(get_flight_mode() == FLIGHT_MODE_WAYPOINT){
		float32 vel_y_error = vel_y_ref - get_nav_y_dot();
		vel_y_error = vel_y_error > VEL_Y_ERR_LIM ? VEL_Y_ERR_LIM : vel_y_error < -VEL_Y_ERR_LIM ? -VEL_Y_ERR_LIM : vel_y_error;
		roll_ref = PID_step(&vel_y_PID, vel_y_error);

		float32 vel_x_error = vel_x_ref - get_nav_x_dot();
		vel_x_error = vel_x_error > VEL_X_ERR_LIM ? VEL_X_ERR_LIM : vel_x_error < -VEL_X_ERR_LIM ? -VEL_X_ERR_LIM : vel_x_error;
		pitch_ref = PID_step(&vel_x_PID, vel_x_error);

		roll_rate_ref = PID_step(&roll_PID, roll_ref - get_roll());
		pitch_rate_ref = PID_step(&pitch_PID, pitch_ref - get_pitch());

		float32 vel_z_error = vel_z_ref - get_NED_z_dot();
		vel_z_error = vel_z_error > VEL_Z_ERR_LIM ? VEL_Z_ERR_LIM : vel_z_error < -VEL_Z_ERR_LIM ? -VEL_Z_ERR_LIM : vel_z_error;
		thr_ref = PID_step(&vel_z_PID, vel_z_error) + VEL_Z_TRIM;
	} else if(get_flight_mode() == FLIGHT_MODE_VELOCITY){
		vel_y_ref *= VEL_Y_SC;
		float32 vel_y_error = vel_y_ref - get_nav_y_dot();
		vel_y_error = vel_y_error > VEL_Y_ERR_LIM ? VEL_Y_ERR_LIM : vel_y_error < -VEL_Y_ERR_LIM ? -VEL_Y_ERR_LIM : vel_y_error;
		roll_ref = PID_step(&vel_y_PID, vel_y_error);

		vel_x_ref *= VEL_X_SC;
		float32 vel_x_error = vel_x_ref - get_nav_x_dot();
		vel_x_error = vel_x_error > VEL_X_ERR_LIM ? VEL_X_ERR_LIM : vel_x_error < -VEL_X_ERR_LIM ? -VEL_X_ERR_LIM : vel_x_error;
		pitch_ref = PID_step(&vel_x_PID, vel_x_error);

		roll_rate_ref = PID_step(&roll_PID, roll_ref - get_roll());
		pitch_rate_ref = PID_step(&pitch_PID, pitch_ref - get_pitch());

		vel_z_ref *= VEL_Z_SC;
		float32 vel_z_error = vel_z_ref - get_NED_z_dot();
		vel_z_error = vel_z_error > VEL_Z_ERR_LIM ? VEL_Z_ERR_LIM : vel_z_error < -VEL_Z_ERR_LIM ? -VEL_Z_ERR_LIM : vel_z_error;
		thr_ref = PID_step(&vel_z_PID, vel_z_error) + VEL_Z_TRIM;

		yaw_rate_ref *= YAW_RATE_SC;
	} else if(get_flight_mode() == FLIGHT_MODE_ANGLE){
		roll_ref *= ROLL_SC;
		pitch_ref *= PITCH_SC;
		roll_rate_ref = PID_step(&roll_PID, roll_ref - get_roll());
		pitch_rate_ref = PID_step(&pitch_PID, pitch_ref - get_pitch());
		yaw_rate_ref *= YAW_RATE_SC;
	} else {
		roll_rate_ref *= ROLL_RATE_SC;
		pitch_rate_ref *= PITCH_RATE_SC;
		yaw_rate_ref *= YAW_RATE_SC;
	}

	PID_step(&roll_rate_PID, roll_rate_ref - get_roll_rate());
	PID_step(&pitch_rate_PID, pitch_rate_ref - get_pitch_rate());
	PID_step(&yaw_rate_PID, yaw_rate_ref - get_yaw_rate());

	float32 m1_cmd = roll_rate_PID.y + pitch_rate_PID.y - yaw_rate_PID.y + thr_ref + 1;
	float32 m2_cmd = - roll_rate_PID.y + pitch_rate_PID.y + yaw_rate_PID.y + thr_ref + 1;
	float32 m3_cmd = -roll_rate_PID.y - pitch_rate_PID.y - yaw_rate_PID.y + thr_ref + 1;
	float32 m4_cmd = roll_rate_PID.y - pitch_rate_PID.y + yaw_rate_PID.y + thr_ref + 1;

	m1_cmd = m1_cmd > IDLE_THR ? m1_cmd < 2 ? m1_cmd : 2 : IDLE_THR;
	m2_cmd = m2_cmd > IDLE_THR ? m2_cmd < 2 ? m2_cmd : 2 : IDLE_THR;
	m3_cmd = m3_cmd > IDLE_THR ? m3_cmd < 2 ? m3_cmd : 2 : IDLE_THR;
	m4_cmd = m4_cmd > IDLE_THR ? m4_cmd < 2 ? m4_cmd : 2 : IDLE_THR;

	if(get_autopilot_state() == AUTOPILOT_ARMED){
		set_PWM_width(1, m1_cmd);
		set_PWM_width(2, m2_cmd);
		set_PWM_width(3, m3_cmd);
		set_PWM_width(4, m4_cmd);
	} else {
		set_PWM_width_all(1);
	}

//	sci_printf("p: %5.2f, q: %5.2f, r: %5.2f, phi: %5.2f, theta: %5.2f, psi: %5.2f\r\n", get_roll_rate()*180/PI, get_pitch_rate()*180/PI, get_yaw_rate()*180/PI, get_roll()*180/PI, get_pitch()*180/PI, get_yaw()*180/PI);

	#if REF_DEBUG == 1
		sci_printf("Roll: %5.2f, Pitch: %5.2f, P: %5.2f, Q: %5.2f, R: %5.2f, Throttle: %5.2f\r\n", roll_ref, pitch_ref, roll_rate_ref, pitch_rate_ref, yaw_rate_ref, thr_ref);
	#endif

	#if MOTOR_DEBUG == 1
		sci_printf("M1: %5.2f, M2: %5.2f, M3: %5.2f, M4: %5.2f\r\n", m1_cmd, m2_cmd, m3_cmd, m4_cmd);
//		sci_printf("Z_alt: %5.2f, Z_dot: %5.2f, Z_ref: %5.2f, M1: %5.2f, M2: %5.2f, M3: %5.2f, M4: %5.2f\r\n", get_NED_z(), get_NED_z_dot(), vel_z_ref, m1_cmd, m2_cmd, m3_cmd, m4_cmd);
	#endif

	#if CONTROLS_DEBUG == 1
		sci_printf("%5.2f,%5.2f,%5.2f,%5.2f,%5.2f,%5.2f,%5.2f,%5.2f,%5.2f,%5.2f,%5.2f,%5.2f,%5.2f,%5.2f\r\n", roll_ref, pitch_ref, yaw_rate_ref, thr_ref, roll, pitch, 0, roll_rate, pitch_rate, yaw_rate, m1_cmd, m2_cmd, m3_cmd, m4_cmd);
	#endif

	#if CONTROLS_DEBUG2 == 1
		sci_printf("%5.3f,%5.3f,%5.3f,%5.3f,%6.3f,%6.3f,%6.3f,%6.3f,%6.3f\r\n",m1_cmd - 1, m2_cmd - 1, m3_cmd - 1, m4_cmd - 1, get_roll(), get_pitch(), get_roll_rate(), get_pitch_rate(), get_yaw_rate());
	#endif
}

void init_PID(PID_controller_t* PID, float32 Ts, float32 P, float32 I, float32 D, float32 N){
	PID->Xi = 0;
	PID->Xd = 0;
	PID->Ts = Ts;
	PID->P = P;
	PID->I = I;
	PID->D = D;
	PID->N = N;
}

float32 PID_step(PID_controller_t* PID, float32 u){
	PID->y = PID->P*u + PID->Xi + PID->N*(PID->D*u - PID->Xd);
	PID->Xi += PID->Ts*u*PID->I;
	PID->Xd += PID->Ts*PID->N*(PID->D*u - PID->Xd);
	return PID->y;
}

void print_PID(){
	sci_printf("%c[2J%c[H", 27, 27);
	sci_printf("Roll Rate \t P: %0.5f, I: %0.5f, D: %0.5f\r\n", PID_params[0], PID_params[1], PID_params[2]);
	sci_printf("Pitch Rate \t P: %0.5f, I: %0.5f, D: %0.5f\r\n", PID_params[3], PID_params[4], PID_params[5]);
	sci_printf("Yaw Rate \t P: %0.5f, I: %0.5f, D: %0.5f\r\n", PID_params[6], PID_params[7], PID_params[8]);
	sci_printf("Roll \t\t P: %0.5f, I: %0.5f, D: %0.5f\r\n", PID_params[9], PID_params[10], PID_params[11]);
	sci_printf("Pitch \t\t P: %0.5f, I: %0.5f, D: %0.5f\r\n", PID_params[12], PID_params[13], PID_params[14]);
}
